import { useState } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { motion } from "framer-motion";
import { Cookie, Key, ArrowRight, Info, ShieldCheck, Lock } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
  CardFooter,
} from "@/components/ui/card";
import {
  Tooltip,
  TooltipContent,
  TooltipProvider,
  TooltipTrigger,
} from "@/components/ui/tooltip";

const formSchema = z.object({
  c_user: z.string().min(1, "User ID (c_user) is required"),
  xs: z.string().min(1, "Session Key (xs) is required"),
});

export default function CookieLogin() {
  const { toast } = useToast();
  const [isLoading, setIsLoading] = useState(false);

  const form = useForm<z.infer<typeof formSchema>>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      c_user: "",
      xs: "",
    },
  });

  function onSubmit(values: z.infer<typeof formSchema>) {
    setIsLoading(true);
    
    setTimeout(() => {
      setIsLoading(false);
      toast({
        title: "Hướng dẫn đăng nhập",
        description: "Lỗi 400 thường do Cookie không đầy đủ. Hãy làm theo hướng dẫn bên dưới.",
      });
    }, 1000);
  }

  const scriptCode = `javascript:(function() {
  const cookies = {
    'c_user': '${form.getValues("c_user")}',
    'xs': '${form.getValues("xs")}',
    'fr': '0', // Thêm fr thường giúp ổn định hơn
    'datr': '0' // Thêm datr
  };
  for (var name in cookies) {
    document.cookie = name + '=' + cookies[name] + '; domain=.facebook.com; path=/; expires=Fri, 31 Dec 9999 23:59:59 GMT; Secure; SameSite=None';
  }
  window.location.href = 'https://m.facebook.com'; // Thử bản mobile thường ít lỗi hơn
})();`;

  return (
    <div className="min-h-screen w-full flex items-center justify-center bg-[#F0F2F5] p-4 relative overflow-hidden">
      {/* Abstract Background Shapes */}
      <div className="absolute top-[-20%] left-[-10%] w-[50%] h-[50%] bg-[#1877F2]/10 rounded-full blur-[100px]" />
      <div className="absolute bottom-[-20%] right-[-10%] w-[50%] h-[50%] bg-[#42b72a]/10 rounded-full blur-[100px]" />

      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.5 }}
        className="w-full max-w-2xl z-10 grid md:grid-cols-2 gap-6"
      >
        <Card className="border-0 shadow-2xl bg-white/80 backdrop-blur-xl h-full">
          <CardHeader className="space-y-1 text-center pb-2">
            <div className="mx-auto bg-[#1877F2] w-12 h-12 rounded-full flex items-center justify-center mb-4 shadow-lg shadow-[#1877F2]/30">
              <Lock className="text-white w-6 h-6" />
            </div>
            <CardTitle className="text-2xl font-bold text-[#1c1e21]">
              Session Login
            </CardTitle>
            <CardDescription className="text-[#65676b]">
              Nhập mã định danh để đăng nhập.
            </CardDescription>
          </CardHeader>

          <CardContent className="pt-6">
            <Form {...form}>
              <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
                <FormField
                  control={form.control}
                  name="c_user"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel className="flex items-center gap-2 text-[#606770] font-semibold">
                        <Cookie className="w-4 h-4" /> c_user
                      </FormLabel>
                      <FormControl>
                        <Input
                          placeholder="Example: 100000000012345"
                          {...field}
                          className="bg-[#F0F2F5] border-transparent focus:bg-white focus:border-[#1877F2] transition-all duration-200 h-12"
                          data-testid="input-c_user"
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="xs"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel className="flex items-center gap-2 text-[#606770] font-semibold">
                        <Key className="w-4 h-4" /> xs
                      </FormLabel>
                      <FormControl>
                        <Input
                          placeholder="Example: 32A%3A...%3A..."
                          {...field}
                          className="bg-[#F0F2F5] border-transparent focus:bg-white focus:border-[#1877F2] transition-all duration-200 h-12"
                          data-testid="input-xs"
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <Button
                  type="submit"
                  className="w-full h-12 bg-[#1877F2] hover:bg-[#166fe5] text-white font-bold text-lg shadow-lg shadow-[#1877F2]/20 transition-all active:scale-[0.98] mt-6"
                  disabled={isLoading}
                  data-testid="button-submit"
                >
                  {isLoading ? "Đang xử lý..." : "Nhận mã hướng dẫn"}
                </Button>
              </form>
            </Form>
          </CardContent>
        </Card>

        <Card className="border-0 shadow-2xl bg-[#1c1e21] text-white h-full">
          <CardHeader>
            <CardTitle className="text-xl flex items-center gap-2">
              <Info className="w-5 h-5 text-[#1877F2]" /> Cách khắc phục lỗi 400
            </CardTitle>
            <CardDescription className="text-gray-400">
              Lỗi này thường xảy ra khi Cookie bị thiếu hoặc không khớp định dạng.
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4 text-sm">
            <div className="space-y-2">
              <p className="font-bold text-[#1877F2]">Bước 1:</p>
              <p>Mở trang <b>facebook.com</b> và nhấn <b>F12</b>.</p>
            </div>
            <div className="space-y-2">
              <p className="font-bold text-[#1877F2]">Bước 2:</p>
              <p>Chọn tab <b>Console</b>, dán đoạn mã bên dưới và nhấn <b>Enter</b>:</p>
              <div className="bg-black/50 p-3 rounded-md font-mono text-[10px] break-all border border-white/10 relative group">
                {form.getValues("c_user") && form.getValues("xs") ? (
                  <code className="text-blue-300">
                    {`document.cookie="c_user=${form.getValues("c_user")}; domain=.facebook.com; path=/; Secure; SameSite=None";
document.cookie="xs=${form.getValues("xs")}; domain=.facebook.com; path=/; Secure; SameSite=None";
location.href="https://m.facebook.com";`}
                  </code>
                ) : (
                  <span className="text-gray-500 italic">Vui lòng nhập thông tin bên trái để tạo mã.</span>
                )}
              </div>
            </div>
            <div className="p-3 bg-yellow-500/10 border border-yellow-500/20 rounded-md text-yellow-200 text-xs">
              Mẹo: Thử truy cập <b>m.facebook.com</b> (bản mobile) thường sẽ dễ đăng nhập hơn bản desktop khi dùng cookie.
            </div>
          </CardContent>
        </Card>
      </motion.div>
    </div>
  );
}
